// Fix downloads in Electron/Nativefier
// Since inject scripts are loaded via require() in preload.js with full Node.js access,
// we can use fs to write files directly to disk and electron.shell to reveal them.
(function () {
    'use strict';

    const fs = require('fs');
    const path = require('path');
    const os = require('os');
    const { shell, dialog } = require('electron').remote || {};

    // Get downloads folder
    function getDownloadsDir() {
        const home = os.homedir();
        const downloadsDir = path.join(home, 'Downloads');
        if (fs.existsSync(downloadsDir)) return downloadsDir;
        return os.tmpdir();
    }

    // Generate unique filename to avoid overwriting
    function getUniqueFilePath(dir, filename) {
        let filePath = path.join(dir, filename);
        if (!fs.existsSync(filePath)) return filePath;

        const ext = path.extname(filename);
        const base = path.basename(filename, ext);
        let counter = 1;
        while (fs.existsSync(filePath)) {
            filePath = path.join(dir, `${base} (${counter})${ext}`);
            counter++;
        }
        return filePath;
    }

    // Convert base64/data URL to Buffer
    function dataUrlToBuffer(dataUrl) {
        const matches = dataUrl.match(/^data:([^;]+);base64,(.+)$/);
        if (matches) {
            return Buffer.from(matches[2], 'base64');
        }
        // Try raw base64
        return Buffer.from(dataUrl, 'base64');
    }

    // Save file and show notification
    function saveFile(dataUrl, filename) {
        try {
            const downloadsDir = getDownloadsDir();
            const filePath = getUniqueFilePath(downloadsDir, filename);
            const buffer = dataUrlToBuffer(dataUrl);

            fs.writeFileSync(filePath, buffer);
            console.log('[Download Fix] File saved to:', filePath);

            // Show the file in Finder/Explorer
            if (shell && shell.showItemInFolder) {
                shell.showItemInFolder(filePath);
            }

            // Show a brief notification using a temporary DOM element
            showSaveNotification(filename, filePath);

            return true;
        } catch (err) {
            console.error('[Download Fix] fs.writeFileSync failed:', err);
            return false;
        }
    }

    // Show a toast notification in the page
    function showSaveNotification(filename, filePath) {
        const toast = document.createElement('div');
        toast.style.cssText = 'position:fixed;bottom:24px;left:50%;transform:translateX(-50%);z-index:999999;background:#1a1a2e;color:white;padding:12px 24px;border-radius:12px;font-family:-apple-system,sans-serif;font-size:14px;box-shadow:0 8px 32px rgba(0,0,0,0.3);display:flex;align-items:center;gap:10px;animation:slideUp 0.3s ease-out;max-width:90vw';
        toast.innerHTML = `
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#4ade80" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M20 6L9 17l-5-5"/>
            </svg>
            <span>Saved: <strong>${filename}</strong></span>
        `;

        const style = document.createElement('style');
        style.textContent = '@keyframes slideUp{from{opacity:0;transform:translateX(-50%) translateY(20px)}to{opacity:1;transform:translateX(-50%) translateY(0)}}';
        document.head.appendChild(style);

        document.body.appendChild(toast);
        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transition = 'opacity 0.3s';
            setTimeout(() => {
                document.body.removeChild(toast);
                document.head.removeChild(style);
            }, 300);
        }, 3000);
    }

    // --- 1. Intercept all clicks on <a download> elements with data:/blob: URLs ---
    document.addEventListener('click', function (e) {
        const anchor = e.target.closest('a[download]');
        if (!anchor) return;

        const href = anchor.href || anchor.getAttribute('href');
        if (!href) return;

        if (href.startsWith('data:') || href.startsWith('blob:')) {
            e.preventDefault();
            e.stopPropagation();

            const filename = anchor.getAttribute('download') || 'download';
            console.log('[Download Fix] Intercepted click:', filename);

            if (href.startsWith('data:')) {
                saveFile(href, filename);
            } else if (href.startsWith('blob:')) {
                fetch(href)
                    .then(res => res.blob())
                    .then(blob => {
                        const reader = new FileReader();
                        reader.onloadend = () => saveFile(reader.result, filename);
                        reader.readAsDataURL(blob);
                    })
                    .catch(err => console.error('[Download Fix] Blob fetch failed:', err));
            }
        }
    }, true);

    // --- 2. Patch programmatic a.click() ---
    const originalClick = HTMLAnchorElement.prototype.click;
    HTMLAnchorElement.prototype.click = function () {
        const href = this.href;
        const downloadAttr = this.getAttribute('download');

        if (downloadAttr !== null && href && (href.startsWith('blob:') || href.startsWith('data:'))) {
            console.log('[Download Fix] Intercepted programmatic click:', downloadAttr);

            if (href.startsWith('data:')) {
                saveFile(href, downloadAttr);
            } else if (href.startsWith('blob:')) {
                fetch(href)
                    .then(res => res.blob())
                    .then(blob => {
                        const reader = new FileReader();
                        reader.onloadend = () => saveFile(reader.result, downloadAttr);
                        reader.readAsDataURL(blob);
                    })
                    .catch(err => {
                        console.error('[Download Fix] Blob fetch failed:', err);
                        originalClick.call(this);
                    });
            }
            return;
        }
        originalClick.call(this);
    };

    // --- 3. Provide window.saveAs (used by PptxGenJS etc.) ---
    window.saveAs = function (blob, filename) {
        console.log('[Download Fix] saveAs called:', filename);
        const reader = new FileReader();
        reader.onloadend = () => saveFile(reader.result, filename || 'download');
        reader.readAsDataURL(blob);
    };

    console.log('[Nativefier] Download fix v4 (Node.js fs) injected. Downloads will save to:', getDownloadsDir());
})();
